# UMBR2014
# Part 07: Means-tested benefit counts
#
# This allocation is done by reference to the experimental OA-level
# claimant counts (for 2012), and by reference to the 2001 Census count
# of *** economically inactive working-age adults ***. The apportionment
# rates are interpolated from 2001 to 2012, then held constant in 2013.
#
# The result is a set of calendar-year four-quarter averages for each
# area and benefit.
# 
# For 2013 only, the (small) number of universal credit claimants from
# the pathfinder LAs are added as JSA claimants, apportioned according
# to the prior total of JSA claimants in that LA's areas.
#
# No special conversions are required for Scotland.
# 
# The file uses:
# - DWP Means-tested benefit counts for JSA, IS, ESA, PC(GE) (2001-2013)
# - DWP experimental Universal Credit statistics
# - 2001 LSOA to 2001 OA benefit allocation weights (see S06)
# - the OA 2001 to LSOA 2011 lookups (see S01)
source("umbr_functions.r")
library(stringr)

# Read all the year files and compile into one set of annual four-quarter averages
bens.read.all.years <- function(yr.files) {
    rbindlist(lapply(yr.files,
                     function(yr.f) {
                         qtr.data <- fread(yr.f)
                         setnames(qtr.data, names(qtr.data), tolower(names(qtr.data)))
                         qtr.data[, year := as.integer(str_extract(date, "20\\d{2}$"))]
                         # 4Q average is sum/4, b/c some benefits have fewer than
                         # 4 quarters' data in some years
                         qtr.data[, list(Count = sum(value) / 4), by=c("geogcode", "year")]
                     } ) )
}

bens.get.one.benefit <- function(ben.label     = "IncSupp",
                                 ben.years     = 2001:2013,
                                 ben.file.name = "incsupp_",
                                 ben.src.dir   = "~/data/Benefits/UMBR/") {
    yr.files <- Sys.glob(sprintf("%s%s*.tsv", ben.src.dir, ben.file.name))
    ben.data <- bens.read.all.years(yr.files)
    dummy.years <- ben.years[! ben.years %in% unique(ben.data$year)]
    dummy.data  <- lapply(dummy.years,
                         function(dummy) {
                             data.table(geogcode = unique(ben.data$geogcode),
                                        year = dummy,
                                        Count = 0) } )
    # Any dummy data for missing years
    ben.data <- rbindlist(list(ben.data, rbindlist(dummy.data)))
    setkeyv(ben.data, c("geogcode", "year"))
    setnames(ben.data, "Count", ben.label)
}

incsupp <- bens.get.one.benefit(ben.label = "IncSupp", ben.file.name="incsupp_")
jsa <- bens.get.one.benefit(ben.label = "JSA", ben.file.name="jsa_")
esa <- bens.get.one.benefit(ben.label = "ESA", ben.file.name="esa_")
pcgc <- bens.get.one.benefit(ben.label = "PCGC", ben.file.name="pcgc_")

# UNIVERSAL CREDIT - 1,178 cases in 2013
uc <- fread("data/uc_claimants-2013-main.csv")
lsoa01.to.la <- fread("data/LSOA01_LSOA11_LAD11_EW_LU.csv")
lsoa01.to.la <- lsoa01.to.la[ LAD11CD %in% uc$LAD11CD,]

# Apportion on share of existing JSA
uc.lsoas <- jsa[ geogcode %in% lsoa01.to.la$LSOA01CD & year == 2013,]
uc.lsoas$LAD11CD <- lsoa01.to.la[ match(uc.lsoas$geogcode, lsoa01.to.la$LSOA01CD), LAD11CD]
uc.lsoas[, JSA.total := sum(JSA), by="LAD11CD"]
uc.lsoas[, UC := uc[match(uc.lsoas$LAD11CD, LAD11CD), UC] * ( JSA / JSA.total) ]
# Copy back to main dataset
jsa[year==2013 & geogcode %in% uc.lsoas$geogcode, JSA := uc.lsoas[,JSA + UC] ]

all.bens <- Reduce(merge, list(jsa, incsupp, esa, pcgc) )
all.bens[, All.MTB := IncSupp + JSA + ESA + PCGC]

# WRITE SCOTTISH ONES
scot.bens <- all.bens[str_detect(geogcode, "^S"),]
write.csv(scot.bens, "working/mtb_scot.csv", row.names=FALSE)

# ENGLAND & WALES:
# re-distribute to 2001 Output Areas & then sum back to 2011 LSOAs
# the weighting / apportionment of LSOA01 workless bens data to OAs
oa.share <- fread("working/lsoa_to_oa-benefits_apportionment.csv")
setnames(all.bens, "geogcode", "LSOA01CD")
oa.share <- merge(oa.share, all.bens, by=c("LSOA01CD", "year"))
oa.share[,All.MTB.share := All.MTB * mtb.share]

# OA to LSOA 11 conversion
oa2lsoa <- fread("working/convert-oa01_to_lsoa11.csv")
# Match ... 
oa.share[,LSOA11CD := oa2lsoa[match(oa.share$OA01CDO, oa2lsoa$OA01CDO), LSOA11CD]]
# ... and total
bens.lsoa11 <- oa.share[,list(All.MTB=sum(All.MTB.share)),
                        by=c("LSOA11CD", "year")]
# See an example of a re-apportioned LSOA in City of London
head(bens.lsoa11[year==2001,])


write.csv(bens.lsoa11, "working/mtb_ew.csv", row.names=FALSE)


